using System;
using System.Collections;
using System.Windows.Forms;

using gov.va.med.vbecs.BOL;

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// Performs UI validation. Provides extended set of validation 
	/// routines specifically for the VBECS business objects.
	/// </summary>
	public class BusinessObjectGuiValidator : GuiValidator
	{
		private string _brokenRulesFormClassName;
		private Hashtable _validationBindings;
		private ArrayList _controlsWithDisabledValidation;

		/// <summary>
		/// Instantiates the validator.
		/// **** 508 Compliance ***
		/// </summary>
		/// <param name="boundErrorProvider">Bound error provider defined on the form validator is attached to.</param>
		/// <param name="brokenRulesFormClassName">Business rules form or class name.</param>
		public BusinessObjectGuiValidator( gov.va.med.vbecs.GUI.controls.VbecsErrorProvider boundErrorProvider, string brokenRulesFormClassName )
			: base( boundErrorProvider )
		{
			if( brokenRulesFormClassName == null )
				throw( new ArgumentNullException( "brokenRulesFormClassName" ) );

			_validationBindings = new Hashtable();
			_controlsWithDisabledValidation = new ArrayList();
			_brokenRulesFormClassName = brokenRulesFormClassName;
		}

		/// <summary>
		/// Binds control to business rules (broken rules). So that if those rules are broken during
		/// validation call the errors will appear for the specified control.
		/// </summary>
		/// <param name="controlToBind">Control to bind.</param>
		/// <param name="rulesNames">Array of rule names.</param>
		public void SetControlBindings( Control controlToBind, params string[] rulesNames )
		{
			if( controlToBind == null )
				throw( new ArgumentNullException( "controlToBind" ) );

			if( rulesNames.Length == 0 )
				_validationBindings.Remove( controlToBind );

			_validationBindings[ controlToBind ] = rulesNames;
		}

		/// <summary>
		/// Performs validation by checking whether business rules are broken or not and 
		/// displaying validation errors for controls. If no bound rules are broken for the control, 
		/// its validation errors are cleared.
		/// </summary>
		/// <param name="objectToValidate">
		///		Business object to validate. If null (nothing selected on form) - 
		///		all validation errors are cleared.
		///	</param>
		public void Validate( BaseBusinessObject objectToValidate )
		{
			if( objectToValidate == null || objectToValidate.BrokenRulesCount == 0 )
			{
				ClearValidationErrorsForAllBoundControls();
				return;
			}

			foreach( Control _controlToBind in _validationBindings.Keys )
			{
				ClearValidationError( _controlToBind );

				if( _controlsWithDisabledValidation.Count != 0 && _controlsWithDisabledValidation.Contains( _controlToBind ) )
					continue;

				foreach( string _ruleName in (string[])_validationBindings[ _controlToBind ] )
				{
					if( !objectToValidate.IsRuleBroken( _ruleName ) )
						continue;

					SetValidationError( _controlToBind, objectToValidate.GetSingleBrokenRuleMessage( _brokenRulesFormClassName, _ruleName ) );
					break;
				}
			}
		}

		/// <summary>
		/// Clears validation errors for all bound controls. 
		/// </summary>
		public void ClearValidationErrorsForAllBoundControls()
		{
			foreach( Control _control in _validationBindings.Keys )
				ClearValidationError( _control );
		}
	}
}
